<?php

class HappyForms_Validation_Messages_Upgrade {
	private $save_action = 'happyforms_save_validation_messages';

	public $save_nonce = 'happyforms-validation-messages-nonce';
	public $messages_option_name = 'happyforms-validation-messages';
	public $validation_messages_controller = '';

	/**
	 * The singleton instance.
	 *
	 * @since 1.0
	 *
	 * @var HappyForms_Form_Controller
	 */
	private static $instance;

	/**
	 * The singleton constructor.
	 *
	 * @since 1.0
	 *
	 * @return HappyForms_Form_Controller
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}

		self::$instance->hook();

		return self::$instance;
	}

	public function hook() {
		$this->validation_messages_controller = happyforms_validation_messages();

		add_action( "wp_ajax_{$this->save_action}", array( $this, 'ajax_save_messages' ) );
		add_filter( 'happyforms_default_validation_messages', array( $this, 'add_messages' ) );
		add_filter( 'happyforms_validation_message', array( $this, 'get_validation_message' ), 10, 2 );
	}

	/**
	 * Adds messages applicable to paid version only.
	 *
	 * @hooked filter `happyforms_default_validation_messages`
	 *
	 * @param array $messages Array of default messages.
	 *
	 * @return array Messages array with new items added.
	 */
	public function add_messages( $messages ) {
		$messages['select_more_choices'] = __( 'Please select more choices', 'happyforms' );
		$messages['select_less_choices'] = __( 'Please select fewer choices', 'happyforms' );
		$messages['file_size_too_big']   = __( 'This file size is too big', 'happyforms' );
		$messages['file_count_exceeded'] = __( 'File count exceeded', 'happyforms' );

		return $messages;
	}

	/**
	 * Wrapper for `get_default_messages()` method in `HappyForms_Validation_Messages`.
	 */
	public function get_default_messages() {
		return $this->validation_messages_controller->get_default_messages();
	}

	/**
	 * Wrapper for `get_message()` method in `HappyForms_Validation_Messages`.
	 */
	public function get_message( $message_key ) {
		return $this->validation_messages_controller->get_message( $message_key );
	}

	/**
	 * Gets custom messages from the database.
	 *
	 * @return array Messages.
	 */
	public function get_custom_messages() {
		$messages = get_option( $this->messages_option_name, array() );

		return $messages;
	}

	/**
	 * Gets all messages (custom + default) as a single array ran through `wp_parse_args`.
	 *
	 * @return array Messages.
	 */
	public function get_all_messages() {
		$messages = get_option( $this->messages_option_name, array() );
		$messages = wp_parse_args( $messages, $this->get_default_messages() );

		return $messages;
	}

	/**
	 * Deletes custom message. Requires message key to be passed.
	 *
	 * @param string $message_key Message key.
	 *
	 * @return void
	 */
	public function delete_custom_message( $message_key ) {
		$messages = $this->get_custom_messages();

		if ( isset( $messages[$message_key] ) ) {
			unset( $messages[$message_key] );
		}

		update_option( $this->messages_option_name, $messages );
	}

	/**
	 * Checks if message key has entry in custom messages array.
	 *
	 * @param string $message_key Message key.
	 *
	 * @return boolean
	 */
	public function is_custom_message( $message_key ) {
		$custom_messages = $this->get_custom_messages();

		if ( isset( $custom_messages[$message_key] ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Save custom messages array to database.
	 *
	 * @param array $messages All messages passed as array.
	 *
	 * @return void
	 */
	public function save_messages( $messages ) {
		$default_messages = $this->get_default_messages();
		$custom_messages = $this->get_custom_messages();

		/**
		 * Running through the array of all messages here and composing an array of custom messages.
		 */
		foreach ( $messages as $message_key => $message ) {
			if ( ! isset( $default_messages[$message_key] ) ) {
				unset( $messages[$message_key] );

				continue;
			}

			$custom_messages[$message_key] = $message;
		}

		update_option( $this->messages_option_name, $custom_messages );
	}

	/**
	 * Gets validation message for `HappyForms_Validation_Messages` when asked.
	 *
	 * @hooked filter `happyforms_validation_message`
	 *
	 * @return string Message.
	 */
	public function get_validation_message( $message, $message_key ) {
		$messages = $this->get_all_messages();

		if ( isset( $messages[$message_key] ) ) {
			return $messages[$message_key];
		}

		return $message;
	}

	/**
	 * Saving routine for Validation messages section on Settings page.
	 *
	 * @hooked action `wp_ajax_happyforms_save_validation_messages`
	 *
	 * @return array WP JSON response.
	 */
	public function ajax_save_messages() {
		if ( ! isset( $_REQUEST[$this->save_nonce] ) ) {
			wp_send_json_error();
		}

		if ( ! check_ajax_referer( $this->save_action, $this->save_nonce, false ) ) {
			wp_send_json_error( __( 'There was an error saving validation messages.', 'happyforms' ) );
		}

		$default_messages = $this->get_default_messages();
		$messages = $this->get_all_messages();
		$messages_to_update = array();

		foreach ( $messages as $message_key => $message ) {
			if ( ! isset( $_REQUEST[$message_key] ) ) {
				continue;
			}

			if ( ! isset( $_REQUEST["custom_{$message_key}"] ) ) {
				if ( $this->is_custom_message( $message_key ) ) {
					$this->delete_custom_message( $message_key );
				}

				continue;
			}

			$message_value = stripslashes( sanitize_text_field( $_REQUEST[$message_key] ) );

			if ( empty( $_REQUEST[$message_key] ) ) {
				wp_send_json_error( __( 'Validation messages cannot be left blank. Please add a message.', 'happyforms' ) );
			}

			$messages_to_update[$message_key] = stripslashes( sanitize_text_field( $message_value ) );
		}

		$this->save_messages( $messages_to_update );

		wp_send_json_success( array(
			'message' => __( 'Changes saved.', 'happyforms' )
		) );
	}
}

if ( ! function_exists( 'happyforms_validation_messages_upgrade' ) ):
/**
 * Get the HappyForms_Validation_Messages_Upgrade class instance.
 *
 * @since 1.0
 *
 * @return HappyForms_Validation_Messages_Upgrade
 */
function happyforms_validation_messages_upgrade() {
	return HappyForms_Validation_Messages_Upgrade::instance();
}

endif;

/**
 * Initialize the HappyForms_Validation_Messages_Upgrade class immediately.
 */
happyforms_validation_messages_upgrade();
