<?php
class HappyForms_Export_Controller {

	private static $instance;

	private $export_import_action = 'happyforms_export_import';

	public $filename = 'happyforms';

	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}

		self::$instance->hook();

		return self::$instance;
	}

	public function hook() {
		add_filter( 'wp_check_filetype_and_ext', array( $this, 'add_xml_mime_type' ), 10, 3 );
		add_action( 'admin_action_happyforms_export_import', array( $this, 'handle_request' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_scripts' ) );
		add_filter( 'happyforms_export_page_method', array( $this, 'set_admin_page_method' ) );
		add_filter( 'happyforms_export_page_url', array( $this, 'set_admin_page_url' ) );
		add_action( 'wp_ajax_happyforms_import_form', array( $this, 'handle_import_request' ) );
	}

	public function add_xml_mime_type( $info, $file, $filename ) {
		if ( isset( $_REQUEST['is_happyforms_export'] ) ) {
			$extension = pathinfo( $filename, PATHINFO_EXTENSION );

			if ( 'xml' !== $extension ) {
				return $info;
			}

			if ( extension_loaded( 'fileinfo' ) ) {
				$finfo = finfo_open( FILEINFO_MIME_TYPE );
				$mime = finfo_file( $finfo, $file );
				finfo_close( $finfo );

				$xml_mimes = array(
					'text/xml', 'application/xml'
				);

				if ( ! in_array( $mime, $xml_mimes ) ) {
					return $info;
				}
			
				$info = array(
					'ext' => $extension,
					'type' => $mime,
				);
			} else {
				$info = array(
					'ext' => $extension,
					'type' => 'text/xml',
				);
			}
		}

		return $info;
	}

	public function set_admin_page_method( $method ) {
		$method = array( $this, 'settings_page' );

		return $method;
	}

	public function set_admin_page_url( $url ) {
		$url = 'happyforms-export';

		return $url;
	}

	public function settings_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( __( 'Sorry, you are not allowed to access this page.', 'happyforms' ) );
		}

		add_filter( 'admin_footer_text', 'happyforms_admin_footer' );

		require_once( happyforms_get_include_folder() . '/templates/admin-export.php' );
	}

	public function admin_enqueue_scripts() {
		$current_screen = get_current_screen();

		if ( 'happyforms_page_happyforms-settings' === $current_screen->id ) {
			wp_register_script(
				'happyforms-export',
				happyforms_get_plugin_url() . 'inc/assets/js/admin/export.js',
				array( 'jquery', 'backbone', 'plupload-handlers' ), HAPPYFORMS_VERSION, true
			);

			wp_enqueue_script( 'happyforms-export' );
		}
	}

	public function handle_request() {
		if ( ! isset( $_REQUEST['happyforms_export_nonce'] ) ) {
			return;
		}

		if ( ! wp_verify_nonce( $_REQUEST['happyforms_export_nonce'], $this->export_import_action ) ) {
			return;
		}

		$action_type = sanitize_text_field( $_REQUEST['action_type'] );
		$form_id = ( isset( $_REQUEST['form_id'] ) ) ? intval( $_REQUEST['form_id'] ) : 0;
		$response = '';

		switch ( $action_type ) {
			case 'export_responses':
				$response = $this->export_responses( $form_id );
			break;

			case 'export_form':
				$response = $this->export_form( $form_id );
			break;

			case 'export_form_responses':
				$response = $this->export_form( $form_id, true );
			break;
		}
	}

	public function export_responses( $form_id ) {
		require_once( happyforms_get_include_folder() . '/classes/class-exporter-csv.php' );

		$exporter = new HappyForms_Exporter_CSV( $form_id, "{$this->filename}.csv" );
		$exporter->export();
	}

	public function export_form( $form_id, $include_responses = false ) {
		require_once( happyforms_get_include_folder() . '/classes/class-exporter-xml.php' );

		$exporter = new HappyForms_Exporter_XML( $form_id, "{$this->filename}.xml" );
		$exporter->export( $include_responses );
	}

	public function handle_import_request() {
		$attachment_id = isset( $_REQUEST['attachment_id'] ) ? intval( $_REQUEST['attachment_id'] ) : 0;

		if ( 0 === $attachment_id ) {
			$message = __( 'Upload failed', 'happyforms' );
			$messages = $this->format_messages( array( $message ) );

			wp_send_json_error( $messages );
		}

		require_once( happyforms_get_include_folder() . '/classes/class-importer-xml.php' );

		$path = get_attached_file( $attachment_id );
		$importer = new HappyForms_Importer_XML();
		$result = $importer->import( $path );

		wp_delete_attachment( $attachment_id, true );

		if ( is_wp_error( $result ) ) {
			$messages = $result->get_error_messages();
			$messages = $this->format_messages( $messages );

			wp_send_json_error( $messages );
		} else {
			$messages = $importer->get_success_messages();
			$messages = $this->format_messages( $messages );

			wp_send_json_success( $messages );
		}
	}

	private function format_messages( $messages ) {
		$messages = array_map( function( $message ) {
			return "<p>$message</p>";
		}, $messages );
		$messages = implode( '', $messages );

		return $messages;
	}

}

if ( ! function_exists( 'happyforms_get_export_controller' ) ):

function happyforms_get_export_controller() {
	return HappyForms_Export_Controller::instance();
}

endif;

happyforms_get_export_controller();
